using System;
using System.IO;
using DynaPDF;

namespace text_extraction2
{
	class TextExtraction
	{
      /*
         Note that the dynapdf.dll must be copied into the output directory or into a
         Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
      */

      // Error callback function.
      // If the function name should not appear at the beginning of the error message set
      // the flag emNoFuncNames (pdf.SetErrorMode(DynaPDF.TErrMode.emNoFuncNames);).
      static int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         Console.Write("{0}\n", System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage));
         Console.Write("\n");
         return 0; // We try to continue if an error occurrs. Any other return value breaks processing.
      }

      // This class extracts the text from a PDF page.
      private static CPDFToText m_TextFile;

      static int parseBeginTemplate(IntPtr Data, IntPtr Object, int Handle, ref TPDFRect BBox, IntPtr Matrix)
      {
         return m_TextFile.BeginTemplate(BBox, Matrix);
      }

      static void parseEndTemplate(IntPtr Data)
      {
         m_TextFile.EndTemplate();
      }

      static void parseMulMatrix(IntPtr Data, IntPtr Object, ref TCTM Matrix)
      {
         m_TextFile.MulMatrix(Matrix);
      }

      static int parseRestoreGraphicState(IntPtr Data)
      {
         m_TextFile.RestoreGState();
         return 0;
      }

      static int parseSaveGraphicState(IntPtr Data)
      {
         return m_TextFile.SaveGState();
      }

      static void parseSetCharSpacing(IntPtr Data, IntPtr Object, double Value)
      {
         m_TextFile.SetCharSpacing(Value);
      }

      static void parseSetFont(IntPtr Data, IntPtr Object, TFontType Type, int Embedded, IntPtr FontName, TFStyle Style, double FontSize, IntPtr IFont)
      {
         m_TextFile.SetFont(FontSize, Type, IFont);
      }

      static void parseSetTextDrawMode(IntPtr Data, IntPtr Object, TDrawMode Mode)
      {
         m_TextFile.SetTextDrawMode(Mode);
      }

      static void parseSetTextScale(IntPtr Data, IntPtr Object, double Value)
      {
         m_TextFile.SetTextScale(Value);
      }

      static void parseSetWordSpacing(IntPtr Data, IntPtr Object, double Value)
      {
         m_TextFile.SetWordSpacing(Value);
      }

      static int parseShowTextArrayW(IntPtr Data, TTextRecordA[] Source, ref TCTM Matrix, TTextRecordW[] Kerning, int Count, double Width, int Decoded)
      {
         return m_TextFile.AddText(Matrix, Source, Kerning, Count, Width, Decoded != 0);
      }

		[STAThread]
		static void Main(string[] args)
		{
		   try
		   {
            CPDF pdf = new CPDF();
            m_TextFile = new CPDFToText(pdf);
            // Error messages and warnings are passed to the callback function.
            pdf.SetOnErrorProc(IntPtr.Zero, new DynaPDF.TErrorProc(PDFError));
            pdf.CreateNewPDF(null); // We create no PDF file in this example

            // External CMaps should always be loaded when processing text from PDF files.
            // See the description of ParseContent() for further information.
            pdf.SetCMapDir(System.IO.Path.GetFullPath("../../../../../Resource/CMap/"), DynaPDF.TLoadCMapFlags.lcmDelayed | DynaPDF.TLoadCMapFlags.lcmRecursive);

            // We avoid the conversion of pages to templates
            pdf.SetImportFlags(DynaPDF.TImportFlags.ifImportAll | DynaPDF.TImportFlags.ifImportAsPage);
            if (pdf.OpenImportFile("../../../../../dynapdf_help.pdf", DynaPDF.TPwdType.ptOpen, null) < 0)
            {
               Console.Write("Input file \"../../../../../../dynapdf_help.pdf\" not found!\n");
               Console.Read();
               return;
            }
            pdf.ImportPDFFile(1, 1.0, 1.0);
            pdf.CloseImportFile();

            // We flatten markup annotations and form fields so that we can extract the text of these objects too.
            pdf.FlattenAnnots(TAnnotFlattenFlags.affMarkupAnnots);
            pdf.FlattenForm();

            DynaPDF.TPDFParseInterface stack = new TPDFParseInterface();
            stack.BeginTemplate       = new TBeginTemplate(parseBeginTemplate);
            stack.EndTemplate         = new TEndTemplate(parseEndTemplate);
            stack.MulMatrix           = new TMulMatrix(parseMulMatrix);
            stack.RestoreGraphicState = new TRestoreGraphicState(parseRestoreGraphicState);
            stack.SaveGraphicState    = new TSaveGraphicState(parseSaveGraphicState);
            stack.SetCharSpacing      = new TSetCharSpacing(parseSetCharSpacing);
            stack.SetFont             = new TSetFont(parseSetFont);
            stack.SetTextScale        = new TSetTextScale(parseSetTextScale);
            stack.SetWordSpacing      = new TSetWordSpacing(parseSetWordSpacing);
            stack.ShowTextArrayW      = new DynaPDF.TShowTextArrayW(parseShowTextArrayW);

            // We write the output file into the current directory.
            String outFile = Directory.GetCurrentDirectory() + "\\out.txt";
            m_TextFile.Open(outFile);

            int i, count = pdf.GetPageCount();
            for (i = 1; i <= count; i++)
            {
               pdf.EditPage(i);
               m_TextFile.Init();
               m_TextFile.WritePageIdentifier(i);
               pdf.ParseContent(IntPtr.Zero, ref stack, DynaPDF.TParseFlags.pfNone);
               pdf.EndPage();
            }
            m_TextFile.Close();
            Console.Write("Text successfully extracted to \"{0}\"!\n", outFile);
            System.Diagnostics.Process p = new System.Diagnostics.Process();
            p.StartInfo.FileName = outFile;
            p.Start();
         }catch(Exception e)
         {
            Console.Write(e.Message);
         }
         Console.Read();
		}
	}
}
